﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

namespace VeteransAffairs.Registries.BusinessAHOBPR
{
    public abstract class EligibilityValidator
    {
        
        public EligibilityValidator(List<string> names, List<string> codes, DateRange dates)
        {
            Names = names;
            Codes = codes;
            EligibleDates = dates;
        }
        public DateRange EligibleDates { get; set; }
        public List<string> Codes { get; set; }
        public List<string> Names { get; set; }

        public bool IsFirstGulfWar { get; set; }

        public bool IsEligible(REGISTRANT_DEPLOYMENT deployment)
        {
            var result = false;

            if (ValidateStartAndEndDates(deployment) && deployment.DEPLOYMENT_COUNTRY != string.Empty)
            {
                result = ValidateCountry(deployment.DEPLOYMENT_COUNTRY);
            }
            return result;
        }

        public bool IsEligible(DateRange deploymentDates, string deploymentCountry)
        {
            var result = false;

            if (ValidateStartAndEndDates(deploymentDates) && deploymentCountry != string.Empty)
            {
                result = ValidateCountry(deploymentCountry);
            }
            return result;
        }

    

        private bool ValidateStartAndEndDates(DateRange deployment)
        {
            return DatesAreEligible(deployment) && StartDateIsNotBeforeEndDate(deployment);
        }

        protected abstract bool DatesAreEligible(DateRange deployment);


        private bool StartDateIsNotBeforeEndDate(DateRange deployment)
        {
            return (deployment.Start <= deployment.End);

        }
        private bool ValidateCountry(string country)
        {
            var result = false;
            var countryCodePattern = new Regex("^[0-9a-zA-Z]{2}$");

            if (countryCodePattern.IsMatch(country))
            {
                result |= Codes.Contains(country, StringComparer.OrdinalIgnoreCase);
            }
            else
            {
                result |= Names.Contains(country, StringComparer.OrdinalIgnoreCase);
            }
            return result;
        }

        private bool ValidateStartAndEndDates(REGISTRANT_DEPLOYMENT deployment)
        {
            return DatesAreNotNull(deployment) && DatesAreEligible(deployment) && StartDateIsNotBeforeEndDate(deployment);
        }

        protected abstract bool DatesAreEligible(REGISTRANT_DEPLOYMENT deployment);


        private bool DatesAreNotNull(REGISTRANT_DEPLOYMENT deployment)
        {
            return (deployment.DEPLOYMENT_START_DATE != null && deployment.DEPLOYMENT_END_DATE != null);
        }

        private bool StartDateIsNotBeforeEndDate(REGISTRANT_DEPLOYMENT deployment)
        {
            return (deployment.DEPLOYMENT_START_DATE <= deployment.DEPLOYMENT_END_DATE);

        }

    }
}